QUnit.test('process url', function(assert) {
  var url2expected = {
    'https://www.cnn.com' : 'https://sur.ly/o/cnn.com/AA000015',
    'https://cnn.com' : 'https://sur.ly/o/cnn.com/AA000015',
    'http://www.cnn.com' : 'http://sur.ly/o/cnn.com/AA000015',
    'http://cnn.com' : 'http://sur.ly/o/cnn.com/AA000015',

    'https://www.cnn.com/' : 'https://sur.ly/o/cnn.com/AA000015',
    'https://cnn.com/' : 'https://sur.ly/o/cnn.com/AA000015',
    'http://www.cnn.com/' : 'http://sur.ly/o/cnn.com/AA000015',
    'http://cnn.com/' : 'http://sur.ly/o/cnn.com/AA000015',

    'https://www.cnn.com/new?ffd' : 'https://sur.ly/o/cnn.com/new%3Fffd/AA000015',
    'https://cnn.com/new?ffd' : 'https://sur.ly/o/cnn.com/new%3Fffd/AA000015',
    'http://www.cnn.com/new?ffd' : 'http://sur.ly/o/cnn.com/new%3Fffd/AA000015',
    'http://cnn.com/new?ffd' : 'http://sur.ly/o/cnn.com/new%3Fffd/AA000015',

    'https://www.cnn.com/new/ss?ffd' : 'https://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA000015',
    'https://cnn.com/new/ss?ffd' : 'https://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA000015',
    'http://www.cnn.com/new/ss?ffd' : 'http://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA000015',
    'http://cnn.com/new/ss?ffd' : 'http://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA000015',

    'http://www.' : 'http://www.',
    'https://www.' : 'https://www.',
    'http://www-www' : 'http://www-www',
    'https://www-www' : 'https://www-www',
    'http://www-www.www' : 'http://sur.ly/o/www-www.www/AA000015',
    'https://www-www.www' : 'https://sur.ly/o/www-www.www/AA000015',
    'http://www.www-www.www' : 'http://sur.ly/o/www-www.www/AA000015',
    'https://www.www-www.www' : 'https://sur.ly/o/www-www.www/AA000015',
    'http://www' : 'http://www',
    'https://www' : 'https://www',
    
    'http://www1.cnn.com' : 'http://sur.ly/o/www1.cnn.com/AA000015',
    'https://www1.cnn.com' : 'https://sur.ly/o/www1.cnn.com/AA000015',

    'http://www1.cnn.com' : 'http://sur.ly/o/www1.cnn.com/AA000015',
    'https://www1.cnn.com' : 'https://sur.ly/o/www1.cnn.com/AA000015',

    'cnn.com' : 'cnn.com',
    'www.cnn.com' : 'www.cnn.com',

    '#' : '#',
    '#anchor' : '#anchor',
    'http://www.some.com/#' : 'http://sur.ly/o/some.com/AA000015',
    'http://www.some.com#hash' : 'http://sur.ly/o/some.com/%23hash/AA000015',
    'http://www.some.com/#hash' : 'http://sur.ly/o/some.com/%23hash/AA000015',
    'http://www.some.com/path/#hash' : 'http://sur.ly/o/some.com/path%2F%23hash/AA000015',
    'http://www.some.com/index.html#hash' : 'http://sur.ly/o/some.com/index.html%23hash/AA000015',
    'http://www.some.com/#hash/path' : 'http://sur.ly/o/some.com/%23hash%2Fpath/AA000015',
    'http://www.some.com/#/path' : 'http://sur.ly/o/some.com/%23%2Fpath/AA000015',
    'http://www.some.com/#!path' : 'http://sur.ly/o/some.com/%23%21path/AA000015',
    'http://www.some.com/?query=foo#hash' : 'http://sur.ly/o/some.com/%3Fquery%3Dfoo%23hash/AA000015',
    'http://www.some.com/some.com#hash' : 'http://sur.ly/o/some.com/some.com%23hash/AA000015',
    'http://www.some.com/path/some.com#hash' : 'http://sur.ly/o/some.com/path%2Fsome.com%23hash/AA000015'
  };

  var surly = new Surly();

  for(var url in url2expected) {
    assert.equal(url2expected[url], surly.processUrl(url), url + ' => ' + url2expected[url]);
  }
});

QUnit.test('process url with toolbarId', function(assert) {
  var url2expected = {
    'https://www.cnn.com' : 'https://sur.ly/o/cnn.com/AA000015',
    'https://cnn.com' : 'https://sur.ly/o/cnn.com/AA000015',
    'http://www.cnn.com' : 'http://sur.ly/o/cnn.com/AA000015',
    'http://cnn.com' : 'http://sur.ly/o/cnn.com/AA000015',

    'https://www.cnn.com/' : 'https://sur.ly/o/cnn.com/AA000015',
    'https://cnn.com/' : 'https://sur.ly/o/cnn.com/AA000015',
    'http://www.cnn.com/' : 'http://sur.ly/o/cnn.com/AA000015',
    'http://cnn.com/' : 'http://sur.ly/o/cnn.com/AA000015',

    'https://www.cnn.com/new?ffd' : 'https://sur.ly/o/cnn.com/new%3Fffd/AA000015',
    'https://cnn.com/new?ffd' : 'https://sur.ly/o/cnn.com/new%3Fffd/AA000015',
    'http://www.cnn.com/new?ffd' : 'http://sur.ly/o/cnn.com/new%3Fffd/AA000015',
    'http://cnn.com/new?ffd' : 'http://sur.ly/o/cnn.com/new%3Fffd/AA000015',

    'https://www.cnn.com/new/ss?ffd' : 'https://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA000015',
    'https://cnn.com/new/ss?ffd' : 'https://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA000015',
    'http://www.cnn.com/new/ss?ffd' : 'http://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA000015',
    'http://cnn.com/new/ss?ffd' : 'http://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA000015',

    'http://www.' : 'http://www.',
    'https://www.' : 'https://www.',
    'http://www-www' : 'http://www-www',
    'https://www-www' : 'https://www-www',
    'http://www-www.www' : 'http://sur.ly/o/www-www.www/AA000015',
    'https://www-www.www' : 'https://sur.ly/o/www-www.www/AA000015',
    'http://www.www-www.www' : 'http://sur.ly/o/www-www.www/AA000015',
    'https://www.www-www.www' : 'https://sur.ly/o/www-www.www/AA000015',
    'http://www' : 'http://www',
    'https://www' : 'https://www',

    'cnn.com' : 'cnn.com',
    'www.cnn.com' : 'www.cnn.com',

    '#' : '#',
    '#anchor' : '#anchor',
    'http://www.some.com/#' : 'http://sur.ly/o/some.com/AA000015',
    'http://www.some.com#hash' : 'http://sur.ly/o/some.com/%23hash/AA000015',
    'http://www.some.com/#hash' : 'http://sur.ly/o/some.com/%23hash/AA000015',
    'http://www.some.com/path/#hash' : 'http://sur.ly/o/some.com/path%2F%23hash/AA000015',
    'http://www.some.com/index.html#hash' : 'http://sur.ly/o/some.com/index.html%23hash/AA000015',
    'http://www.some.com/#hash/path' : 'http://sur.ly/o/some.com/%23hash%2Fpath/AA000015',
    'http://www.some.com/#/path' : 'http://sur.ly/o/some.com/%23%2Fpath/AA000015',
    'http://www.some.com/#!path' : 'http://sur.ly/o/some.com/%23%21path/AA000015',
    'http://www.some.com/?query=foo#hash' : 'http://sur.ly/o/some.com/%3Fquery%3Dfoo%23hash/AA000015',
    'http://www.some.com/some.com#hash' : 'http://sur.ly/o/some.com/some.com%23hash/AA000015',
    'http://www.some.com/path/some.com#hash' : 'http://sur.ly/o/some.com/path%2Fsome.com%23hash/AA000015'
  };

  var surly = new Surly('AA000015');

  for(var url in url2expected) {
    assert.equal(url2expected[url], surly.processUrl(url), url + ' => ' + url2expected[url]);
  }
});

QUnit.test('out panel: process url', function(assert) {
  var url2expected = {
    'https://www.cnn.com' : ['http://sub.domain.com/s/cnn.com/', 'http://wwwebanything.subdomian.com/s/cnn.com/'],
    'https://cnn.com' : ['http://sub.domain.com/s/cnn.com/', 'http://wwwebanything.subdomian.com/s/cnn.com/'],
    'http://www.cnn.com' : ['http://sub.domain.com/cnn.com/', 'http://wwwebanything.subdomian.com/cnn.com/'],
    'http://cnn.com' : ['http://sub.domain.com/cnn.com/', 'http://wwwebanything.subdomian.com/cnn.com/'],

    'https://www.cnn.com/' : ['http://sub.domain.com/s/cnn.com/', 'http://wwwebanything.subdomian.com/s/cnn.com/'],
    'https://cnn.com/' : ['http://sub.domain.com/s/cnn.com/', 'http://wwwebanything.subdomian.com/s/cnn.com/'],
    'http://www.cnn.com/' : ['http://sub.domain.com/cnn.com/', 'http://wwwebanything.subdomian.com/cnn.com/'],
    'http://cnn.com/' : ['http://sub.domain.com/cnn.com/', 'http://wwwebanything.subdomian.com/cnn.com/'],

    'https://www.cnn.com/new?ffd' : ['http://sub.domain.com/s/cnn.com/new%3Fffd', 'http://wwwebanything.subdomian.com/s/cnn.com/new%3Fffd'],
    'https://cnn.com/new?ffd' : ['http://sub.domain.com/s/cnn.com/new%3Fffd', 'http://wwwebanything.subdomian.com/s/cnn.com/new%3Fffd'],
    'http://www.cnn.com/new?ffd' : ['http://sub.domain.com/cnn.com/new%3Fffd', 'http://wwwebanything.subdomian.com/cnn.com/new%3Fffd'],
    'http://cnn.com/new?ffd' : ['http://sub.domain.com/cnn.com/new%3Fffd', 'http://wwwebanything.subdomian.com/cnn.com/new%3Fffd'],

    'https://www.cnn.com/new/ss?ffd' : ['http://sub.domain.com/s/cnn.com/new%2Fss%3Fffd', 'http://wwwebanything.subdomian.com/s/cnn.com/new%2Fss%3Fffd'],
    'https://cnn.com/new/ss?ffd' : ['http://sub.domain.com/s/cnn.com/new%2Fss%3Fffd', 'http://wwwebanything.subdomian.com/s/cnn.com/new%2Fss%3Fffd'],
    'http://www.cnn.com/new/ss?ffd' : ['http://sub.domain.com/cnn.com/new%2Fss%3Fffd', 'http://wwwebanything.subdomian.com/cnn.com/new%2Fss%3Fffd'],
    'http://cnn.com/new/ss?ffd' : ['http://sub.domain.com/cnn.com/new%2Fss%3Fffd', 'http://wwwebanything.subdomian.com/cnn.com/new%2Fss%3Fffd'],

    'http://www' : ['http://www', 'http://www'],
    'https://www' : ['https://www', 'https://www'],
    'http://www.' : ['http://www.', 'http://www.'],
    'https://www.' : ['https://www.', 'https://www.'],
    'http://www-www' : ['http://www-www', 'http://www-www'],
    'https://www-www' : ['https://www-www', 'https://www-www'],
    'http://www-www.www' : ['http://sub.domain.com/www-www.www/', 'http://wwwebanything.subdomian.com/www-www.www/'],
    'https://www-www.www' : ['http://sub.domain.com/s/www-www.www/', 'http://wwwebanything.subdomian.com/s/www-www.www/'],
    'http://www.www-www.www' : ['http://sub.domain.com/www-www.www/', 'http://wwwebanything.subdomian.com/www-www.www/'],
    'https://www.www-www.www' : ['http://sub.domain.com/s/www-www.www/', 'http://wwwebanything.subdomian.com/s/www-www.www/'],
    
    'http://www1.cnn.com' : ['http://sub.domain.com/www1.cnn.com/', 'http://wwwebanything.subdomian.com/www1.cnn.com/'],
    'https://www1.cnn.com' : ['http://sub.domain.com/s/www1.cnn.com/', 'http://wwwebanything.subdomian.com/s/www1.cnn.com/'],
 
    'http://www1.cnn.com' : ['http://sub.domain.com/www1.cnn.com/', 'http://wwwebanything.subdomian.com/www1.cnn.com/'],
    'https://www1.cnn.com' : ['http://sub.domain.com/s/www1.cnn.com/', 'http://wwwebanything.subdomian.com/s/www1.cnn.com/'],

    'cnn.com' : ['cnn.com', 'cnn.com'],
    'www.cnn.com' : ['www.cnn.com', 'www.cnn.com'],

    '#' : ['#', '#'],
    '#anchor' : ['#anchor', '#anchor'],
    'http://www.some.com/#' : ['http://sub.domain.com/some.com/', 'http://wwwebanything.subdomian.com/some.com/'],
    'http://www.some.com#hash' : ['http://sub.domain.com/some.com/%23hash', 'http://wwwebanything.subdomian.com/some.com/%23hash'],
    'http://www.some.com/#hash' : ['http://sub.domain.com/some.com/%23hash', 'http://wwwebanything.subdomian.com/some.com/%23hash'],
    'http://www.some.com/path/#hash' : ['http://sub.domain.com/some.com/path%2F%23hash', 'http://wwwebanything.subdomian.com/some.com/path%2F%23hash'],
    'http://www.some.com/index.html#hash' : ['http://sub.domain.com/some.com/index.html%23hash', 'http://wwwebanything.subdomian.com/some.com/index.html%23hash'],
    'http://www.some.com/#hash/path' : ['http://sub.domain.com/some.com/%23hash%2Fpath', 'http://wwwebanything.subdomian.com/some.com/%23hash%2Fpath'],
    'http://www.some.com/#/path' : ['http://sub.domain.com/some.com/%23%2Fpath', 'http://wwwebanything.subdomian.com/some.com/%23%2Fpath'],
    'http://www.some.com/#!path' : ['http://sub.domain.com/some.com/%23%21path', 'http://wwwebanything.subdomian.com/some.com/%23%21path'],
    'http://www.some.com/?query=foo#hash' : ['http://sub.domain.com/some.com/%3Fquery%3Dfoo%23hash', 'http://wwwebanything.subdomian.com/some.com/%3Fquery%3Dfoo%23hash'],
    'http://www.some.com/some.com#hash' : ['http://sub.domain.com/some.com/some.com%23hash', 'http://wwwebanything.subdomian.com/some.com/some.com%23hash'],
    'http://www.some.com/path/some.com#hash' : ['http://sub.domain.com/some.com/path%2Fsome.com%23hash', 'http://wwwebanything.subdomian.com/some.com/path%2Fsome.com%23hash']
  };

  var surly = new Surly();
  surly.setPanelHost('sub.domain.com');

  for(var url in url2expected) {
    assert.equal(url2expected[url][0], surly.processUrl(url), url + ' => ' + url2expected[url][0]);
  }

  surly.setPanelHost('wwwebanything.subdomian.com');

  for(var url in url2expected) {
    assert.equal(url2expected[url][1], surly.processUrl(url), url + ' => ' + url2expected[url][1]);
  }
});

QUnit.test('out panel: process url with toolbarId', function(assert) {
  var url2expected = {
    'https://www.cnn.com' : ['http://sub.domain.com/s/cnn.com/', 'http://wwwebanything.subdomian.com/s/cnn.com/'],
    'https://cnn.com' : ['http://sub.domain.com/s/cnn.com/', 'http://wwwebanything.subdomian.com/s/cnn.com/'],
    'http://www.cnn.com' : ['http://sub.domain.com/cnn.com/', 'http://wwwebanything.subdomian.com/cnn.com/'],
    'http://cnn.com' : ['http://sub.domain.com/cnn.com/', 'http://wwwebanything.subdomian.com/cnn.com/'],

    'https://www.cnn.com/' : ['http://sub.domain.com/s/cnn.com/', 'http://wwwebanything.subdomian.com/s/cnn.com/'],
    'https://cnn.com/' : ['http://sub.domain.com/s/cnn.com/', 'http://wwwebanything.subdomian.com/s/cnn.com/'],
    'http://www.cnn.com/' : ['http://sub.domain.com/cnn.com/', 'http://wwwebanything.subdomian.com/cnn.com/'],
    'http://cnn.com/' : ['http://sub.domain.com/cnn.com/', 'http://wwwebanything.subdomian.com/cnn.com/'],

    'https://www.cnn.com/new?ffd' : ['http://sub.domain.com/s/cnn.com/new%3Fffd', 'http://wwwebanything.subdomian.com/s/cnn.com/new%3Fffd'],
    'https://cnn.com/new?ffd' : ['http://sub.domain.com/s/cnn.com/new%3Fffd', 'http://wwwebanything.subdomian.com/s/cnn.com/new%3Fffd'],
    'http://www.cnn.com/new?ffd' : ['http://sub.domain.com/cnn.com/new%3Fffd', 'http://wwwebanything.subdomian.com/cnn.com/new%3Fffd'],
    'http://cnn.com/new?ffd' : ['http://sub.domain.com/cnn.com/new%3Fffd', 'http://wwwebanything.subdomian.com/cnn.com/new%3Fffd'],

    'https://www.cnn.com/new/ss?ffd' : ['http://sub.domain.com/s/cnn.com/new%2Fss%3Fffd', 'http://wwwebanything.subdomian.com/s/cnn.com/new%2Fss%3Fffd'],
    'https://cnn.com/new/ss?ffd' : ['http://sub.domain.com/s/cnn.com/new%2Fss%3Fffd', 'http://wwwebanything.subdomian.com/s/cnn.com/new%2Fss%3Fffd'],
    'http://www.cnn.com/new/ss?ffd' : ['http://sub.domain.com/cnn.com/new%2Fss%3Fffd', 'http://wwwebanything.subdomian.com/cnn.com/new%2Fss%3Fffd'],
    'http://cnn.com/new/ss?ffd' : ['http://sub.domain.com/cnn.com/new%2Fss%3Fffd', 'http://wwwebanything.subdomian.com/cnn.com/new%2Fss%3Fffd'],

    'http://www' : ['http://www', 'http://www'],
    'https://www' : ['https://www', 'https://www'],
    'http://www.' : ['http://www.', 'http://www.'],
    'https://www.' : ['https://www.', 'https://www.'],
    'http://www-www' : ['http://www-www', 'http://www-www'],
    'https://www-www' : ['https://www-www', 'https://www-www'],
    'http://www-www.www' : ['http://sub.domain.com/www-www.www/', 'http://wwwebanything.subdomian.com/www-www.www/'],
    'https://www-www.www' : ['http://sub.domain.com/s/www-www.www/', 'http://wwwebanything.subdomian.com/s/www-www.www/'],
    'http://www.www-www.www' : ['http://sub.domain.com/www-www.www/', 'http://wwwebanything.subdomian.com/www-www.www/'],
    'https://www.www-www.www' : ['http://sub.domain.com/s/www-www.www/', 'http://wwwebanything.subdomian.com/s/www-www.www/'],
    
    'http://www1.cnn.com' : ['http://sub.domain.com/www1.cnn.com/', 'http://wwwebanything.subdomian.com/www1.cnn.com/'],
    'https://www1.cnn.com' : ['http://sub.domain.com/s/www1.cnn.com/', 'http://wwwebanything.subdomian.com/s/www1.cnn.com/'],
 
    'http://www1.cnn.com' : ['http://sub.domain.com/www1.cnn.com/', 'http://wwwebanything.subdomian.com/www1.cnn.com/'],
    'https://www1.cnn.com' : ['http://sub.domain.com/s/www1.cnn.com/', 'http://wwwebanything.subdomian.com/s/www1.cnn.com/'],

    'cnn.com' : ['cnn.com', 'cnn.com'],
    'www.cnn.com' : ['www.cnn.com', 'www.cnn.com'],

    '#' : ['#', '#'],
    '#anchor' : ['#anchor', '#anchor'],
    'http://www.some.com/#' : ['http://sub.domain.com/some.com/', 'http://wwwebanything.subdomian.com/some.com/'],
    'http://www.some.com#hash' : ['http://sub.domain.com/some.com/%23hash', 'http://wwwebanything.subdomian.com/some.com/%23hash'],
    'http://www.some.com/#hash' : ['http://sub.domain.com/some.com/%23hash', 'http://wwwebanything.subdomian.com/some.com/%23hash'],
    'http://www.some.com/path/#hash' : ['http://sub.domain.com/some.com/path%2F%23hash', 'http://wwwebanything.subdomian.com/some.com/path%2F%23hash'],
    'http://www.some.com/index.html#hash' : ['http://sub.domain.com/some.com/index.html%23hash', 'http://wwwebanything.subdomian.com/some.com/index.html%23hash'],
    'http://www.some.com/#hash/path' : ['http://sub.domain.com/some.com/%23hash%2Fpath', 'http://wwwebanything.subdomian.com/some.com/%23hash%2Fpath'],
    'http://www.some.com/#/path' : ['http://sub.domain.com/some.com/%23%2Fpath', 'http://wwwebanything.subdomian.com/some.com/%23%2Fpath'],
    'http://www.some.com/#!path' : ['http://sub.domain.com/some.com/%23%21path', 'http://wwwebanything.subdomian.com/some.com/%23%21path'],
    'http://www.some.com/?query=foo#hash' : ['http://sub.domain.com/some.com/%3Fquery%3Dfoo%23hash', 'http://wwwebanything.subdomian.com/some.com/%3Fquery%3Dfoo%23hash'],
    'http://www.some.com/some.com#hash' : ['http://sub.domain.com/some.com/some.com%23hash', 'http://wwwebanything.subdomian.com/some.com/some.com%23hash'],
    'http://www.some.com/path/some.com#hash' : ['http://sub.domain.com/some.com/path%2Fsome.com%23hash', 'http://wwwebanything.subdomian.com/some.com/path%2Fsome.com%23hash']
  };

  var surly = new Surly('AA000015');
  surly.setPanelHost('http://sub.domain.com');

  for(var url in url2expected) {
    assert.equal(url2expected[url][0], surly.processUrl(url), url + ' => ' + url2expected[url][0]);
  }

  surly.setPanelHost('http://wwwebanything.subdomian.com');

  for(var url in url2expected) {
    assert.equal(url2expected[url][1], surly.processUrl(url), url + ' => ' + url2expected[url][1]);
  }
});

QUnit.test('process multiple urls', function(assert) {
  var url2expected = {
    'https://www.cnn.com' : 'https://sur.ly/o/cnn.com/AA000015',
    'https://cnn.com' : 'https://sur.ly/o/cnn.com/AA000015',
    'http://www.cnn.com' : 'http://sur.ly/o/cnn.com/AA000015',
    'http://cnn.com' : 'http://sur.ly/o/cnn.com/AA000015',

    'https://www.cnn.com/' : 'https://sur.ly/o/cnn.com/AA000015',
    'https://cnn.com/' : 'https://sur.ly/o/cnn.com/AA000015',
    'http://www.cnn.com/' : 'http://sur.ly/o/cnn.com/AA000015',
    'http://cnn.com/' : 'http://sur.ly/o/cnn.com/AA000015',

    'https://www.cnn.com/new?ffd' : 'https://sur.ly/o/cnn.com/new%3Fffd/AA000015',
    'https://cnn.com/new?ffd' : 'https://sur.ly/o/cnn.com/new%3Fffd/AA000015',
    'http://www.cnn.com/new?ffd' : 'http://sur.ly/o/cnn.com/new%3Fffd/AA000015',
    'http://cnn.com/new?ffd' : 'http://sur.ly/o/cnn.com/new%3Fffd/AA000015',

    'https://www.cnn.com/new/ss?ffd' : 'https://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA000015',
    'https://cnn.com/new/ss?ffd' : 'https://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA000015',
    'http://www.cnn.com/new/ss?ffd' : 'http://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA000015',
    'http://cnn.com/new/ss?ffd' : 'http://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA000015',

    'http://www.' : 'http://www.',
    'https://www.' : 'https://www.',
    'http://www-www' : 'http://www-www',
    'https://www-www' : 'https://www-www',
    'http://www-www.www' : 'http://sur.ly/o/www-www.www/AA000015',
    'https://www-www.www' : 'https://sur.ly/o/www-www.www/AA000015',
    'http://www.www-www.www' : 'http://sur.ly/o/www-www.www/AA000015',
    'https://www.www-www.www' : 'https://sur.ly/o/www-www.www/AA000015',
    'http://www' : 'http://www',
    'https://www' : 'https://www',

    'cnn.com' : 'cnn.com',
    'www.cnn.com' : 'www.cnn.com',

    '#' : '#',
    '#anchor' : '#anchor',
    'http://www.some.com/#' : 'http://sur.ly/o/some.com/AA000015',
    'http://www.some.com#hash' : 'http://sur.ly/o/some.com/%23hash/AA000015',
    'http://www.some.com/#hash' : 'http://sur.ly/o/some.com/%23hash/AA000015',
    'http://www.some.com/path/#hash' : 'http://sur.ly/o/some.com/path%2F%23hash/AA000015',
    'http://www.some.com/index.html#hash' : 'http://sur.ly/o/some.com/index.html%23hash/AA000015',
    'http://www.some.com/#hash/path' : 'http://sur.ly/o/some.com/%23hash%2Fpath/AA000015',
    'http://www.some.com/#/path' : 'http://sur.ly/o/some.com/%23%2Fpath/AA000015',
    'http://www.some.com/#!path' : 'http://sur.ly/o/some.com/%23%21path/AA000015',
    'http://www.some.com/?query=foo#hash' : 'http://sur.ly/o/some.com/%3Fquery%3Dfoo%23hash/AA000015',
    'http://www.some.com/some.com#hash' : 'http://sur.ly/o/some.com/some.com%23hash/AA000015',
    'http://www.some.com/path/some.com#hash' : 'http://sur.ly/o/some.com/path%2Fsome.com%23hash/AA000015'
  };

  var surly = new Surly();

  for(var url in url2expected) {
    assert.deepEqual([url2expected[url]], surly.processMultipleUrls([url]), '[' + url + ']' + ' => ' + '[' + url2expected[url] + ']');
  }
});

QUnit.test('process multiple urls with toolbarId', function(assert) {  
  var url2expected = {
    'https://www.cnn.com' : 'https://sur.ly/o/cnn.com/AA000015',
    'https://cnn.com' : 'https://sur.ly/o/cnn.com/AA000015',
    'http://www.cnn.com' : 'http://sur.ly/o/cnn.com/AA000015',
    'http://cnn.com' : 'http://sur.ly/o/cnn.com/AA000015',

    'https://www.cnn.com/' : 'https://sur.ly/o/cnn.com/AA000015',
    'https://cnn.com/' : 'https://sur.ly/o/cnn.com/AA000015',
    'http://www.cnn.com/' : 'http://sur.ly/o/cnn.com/AA000015',
    'http://cnn.com/' : 'http://sur.ly/o/cnn.com/AA000015',

    'https://www.cnn.com/new?ffd' : 'https://sur.ly/o/cnn.com/new%3Fffd/AA000015',
    'https://cnn.com/new?ffd' : 'https://sur.ly/o/cnn.com/new%3Fffd/AA000015',
    'http://www.cnn.com/new?ffd' : 'http://sur.ly/o/cnn.com/new%3Fffd/AA000015',
    'http://cnn.com/new?ffd' : 'http://sur.ly/o/cnn.com/new%3Fffd/AA000015',

    'https://www.cnn.com/new/ss?ffd' : 'https://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA000015',
    'https://cnn.com/new/ss?ffd' : 'https://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA000015',
    'http://www.cnn.com/new/ss?ffd' : 'http://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA000015',
    'http://cnn.com/new/ss?ffd' : 'http://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA000015',

    'http://www.' : 'http://www.',
    'https://www.' : 'https://www.',
    'http://www-www' : 'http://www-www',
    'https://www-www' : 'https://www-www',
    'http://www-www.www' : 'http://sur.ly/o/www-www.www/AA000015',
    'https://www-www.www' : 'https://sur.ly/o/www-www.www/AA000015',
    'http://www.www-www.www' : 'http://sur.ly/o/www-www.www/AA000015',
    'https://www.www-www.www' : 'https://sur.ly/o/www-www.www/AA000015',
    'http://www' : 'http://www',
    'https://www' : 'https://www',

    'cnn.com' : 'cnn.com',
    'www.cnn.com' : 'www.cnn.com',

    '#' : '#',
    '#anchor' : '#anchor',
    'http://www.some.com/#' : 'http://sur.ly/o/some.com/AA000015',
    'http://www.some.com#hash' : 'http://sur.ly/o/some.com/%23hash/AA000015',
    'http://www.some.com/#hash' : 'http://sur.ly/o/some.com/%23hash/AA000015',
    'http://www.some.com/path/#hash' : 'http://sur.ly/o/some.com/path%2F%23hash/AA000015',
    'http://www.some.com/index.html#hash' : 'http://sur.ly/o/some.com/index.html%23hash/AA000015',
    'http://www.some.com/#hash/path' : 'http://sur.ly/o/some.com/%23hash%2Fpath/AA000015',
    'http://www.some.com/#/path' : 'http://sur.ly/o/some.com/%23%2Fpath/AA000015',
    'http://www.some.com/#!path' : 'http://sur.ly/o/some.com/%23%21path/AA000015',
    'http://www.some.com/?query=foo#hash' : 'http://sur.ly/o/some.com/%3Fquery%3Dfoo%23hash/AA000015',
    'http://www.some.com/some.com#hash' : 'http://sur.ly/o/some.com/some.com%23hash/AA000015',
    'http://www.some.com/path/some.com#hash' : 'http://sur.ly/o/some.com/path%2Fsome.com%23hash/AA000015'
  };

  var surly = new Surly('AA000015');

  for(var url in url2expected) {
    assert.deepEqual([url2expected[url]], surly.processMultipleUrls([url]), '[' + url + ']' + ' => ' + '[' + url2expected[url] + ']');
  }
});

QUnit.test('out panel: process multiple urls', function(assert) {
  var url2expected = {
    'https://www.cnn.com' : ['http://sub.domain.com/s/cnn.com/', 'http://wwwebanything.subdomian.com/s/cnn.com/'],
    'https://cnn.com' : ['http://sub.domain.com/s/cnn.com/', 'http://wwwebanything.subdomian.com/s/cnn.com/'],
    'http://www.cnn.com' : ['http://sub.domain.com/cnn.com/', 'http://wwwebanything.subdomian.com/cnn.com/'],
    'http://cnn.com' : ['http://sub.domain.com/cnn.com/', 'http://wwwebanything.subdomian.com/cnn.com/'],

    'https://www.cnn.com/' : ['http://sub.domain.com/s/cnn.com/', 'http://wwwebanything.subdomian.com/s/cnn.com/'],
    'https://cnn.com/' : ['http://sub.domain.com/s/cnn.com/', 'http://wwwebanything.subdomian.com/s/cnn.com/'],
    'http://www.cnn.com/' : ['http://sub.domain.com/cnn.com/', 'http://wwwebanything.subdomian.com/cnn.com/'],
    'http://cnn.com/' : ['http://sub.domain.com/cnn.com/', 'http://wwwebanything.subdomian.com/cnn.com/'],

    'https://www.cnn.com/new?ffd' : ['http://sub.domain.com/s/cnn.com/new%3Fffd', 'http://wwwebanything.subdomian.com/s/cnn.com/new%3Fffd'],
    'https://cnn.com/new?ffd' : ['http://sub.domain.com/s/cnn.com/new%3Fffd', 'http://wwwebanything.subdomian.com/s/cnn.com/new%3Fffd'],
    'http://www.cnn.com/new?ffd' : ['http://sub.domain.com/cnn.com/new%3Fffd', 'http://wwwebanything.subdomian.com/cnn.com/new%3Fffd'],
    'http://cnn.com/new?ffd' : ['http://sub.domain.com/cnn.com/new%3Fffd', 'http://wwwebanything.subdomian.com/cnn.com/new%3Fffd'],

    'https://www.cnn.com/new/ss?ffd' : ['http://sub.domain.com/s/cnn.com/new%2Fss%3Fffd', 'http://wwwebanything.subdomian.com/s/cnn.com/new%2Fss%3Fffd'],
    'https://cnn.com/new/ss?ffd' : ['http://sub.domain.com/s/cnn.com/new%2Fss%3Fffd', 'http://wwwebanything.subdomian.com/s/cnn.com/new%2Fss%3Fffd'],
    'http://www.cnn.com/new/ss?ffd' : ['http://sub.domain.com/cnn.com/new%2Fss%3Fffd', 'http://wwwebanything.subdomian.com/cnn.com/new%2Fss%3Fffd'],
    'http://cnn.com/new/ss?ffd' : ['http://sub.domain.com/cnn.com/new%2Fss%3Fffd', 'http://wwwebanything.subdomian.com/cnn.com/new%2Fss%3Fffd'],

    'http://www' : ['http://www', 'http://www'],
    'https://www' : ['https://www', 'https://www'],
    'http://www.' : ['http://www.', 'http://www.'],
    'https://www.' : ['https://www.', 'https://www.'],
    'http://www-www' : ['http://www-www', 'http://www-www'],
    'https://www-www' : ['https://www-www', 'https://www-www'],
    'http://www-www.www' : ['http://sub.domain.com/www-www.www/', 'http://wwwebanything.subdomian.com/www-www.www/'],
    'https://www-www.www' : ['http://sub.domain.com/s/www-www.www/', 'http://wwwebanything.subdomian.com/s/www-www.www/'],
    'http://www.www-www.www' : ['http://sub.domain.com/www-www.www/', 'http://wwwebanything.subdomian.com/www-www.www/'],
    'https://www.www-www.www' : ['http://sub.domain.com/s/www-www.www/', 'http://wwwebanything.subdomian.com/s/www-www.www/'],
    
    'http://www1.cnn.com' : ['http://sub.domain.com/www1.cnn.com/', 'http://wwwebanything.subdomian.com/www1.cnn.com/'],
    'https://www1.cnn.com' : ['http://sub.domain.com/s/www1.cnn.com/', 'http://wwwebanything.subdomian.com/s/www1.cnn.com/'],
 
    'http://www1.cnn.com' : ['http://sub.domain.com/www1.cnn.com/', 'http://wwwebanything.subdomian.com/www1.cnn.com/'],
    'https://www1.cnn.com' : ['http://sub.domain.com/s/www1.cnn.com/', 'http://wwwebanything.subdomian.com/s/www1.cnn.com/'],

    'cnn.com' : ['cnn.com', 'cnn.com'],
    'www.cnn.com' : ['www.cnn.com', 'www.cnn.com'],

    '#' : ['#', '#'],
    '#anchor' : ['#anchor', '#anchor'],
    'http://www.some.com/#' : ['http://sub.domain.com/some.com/', 'http://wwwebanything.subdomian.com/some.com/'],
    'http://www.some.com#hash' : ['http://sub.domain.com/some.com/%23hash', 'http://wwwebanything.subdomian.com/some.com/%23hash'],
    'http://www.some.com/#hash' : ['http://sub.domain.com/some.com/%23hash', 'http://wwwebanything.subdomian.com/some.com/%23hash'],
    'http://www.some.com/path/#hash' : ['http://sub.domain.com/some.com/path%2F%23hash', 'http://wwwebanything.subdomian.com/some.com/path%2F%23hash'],
    'http://www.some.com/index.html#hash' : ['http://sub.domain.com/some.com/index.html%23hash', 'http://wwwebanything.subdomian.com/some.com/index.html%23hash'],
    'http://www.some.com/#hash/path' : ['http://sub.domain.com/some.com/%23hash%2Fpath', 'http://wwwebanything.subdomian.com/some.com/%23hash%2Fpath'],
    'http://www.some.com/#/path' : ['http://sub.domain.com/some.com/%23%2Fpath', 'http://wwwebanything.subdomian.com/some.com/%23%2Fpath'],
    'http://www.some.com/#!path' : ['http://sub.domain.com/some.com/%23%21path', 'http://wwwebanything.subdomian.com/some.com/%23%21path'],
    'http://www.some.com/?query=foo#hash' : ['http://sub.domain.com/some.com/%3Fquery%3Dfoo%23hash', 'http://wwwebanything.subdomian.com/some.com/%3Fquery%3Dfoo%23hash'],
    'http://www.some.com/some.com#hash' : ['http://sub.domain.com/some.com/some.com%23hash', 'http://wwwebanything.subdomian.com/some.com/some.com%23hash'],
    'http://www.some.com/path/some.com#hash' : ['http://sub.domain.com/some.com/path%2Fsome.com%23hash', 'http://wwwebanything.subdomian.com/some.com/path%2Fsome.com%23hash']
  };

  var surly = new Surly();
  surly.setPanelHost('https://sub.domain.com');

  for(var url in url2expected) {
    assert.deepEqual([url2expected[url][0]], surly.processMultipleUrls([url]), '[' + url + ']' + ' => ' + '[' + url2expected[url][0] + ']');
  }

  surly.setPanelHost('https://wwwebanything.subdomian.com');

  for(var url in url2expected) {
    assert.deepEqual([url2expected[url][1]], surly.processMultipleUrls([url]), '[' + url + ']' + ' => ' + '[' + url2expected[url][1] + ']');
  }
});

QUnit.test('out panel: process multiple urls with toolbarId', function(assert) {
  var url2expected = {
    'https://www.cnn.com' : ['http://sub.domain.com/s/cnn.com/', 'http://wwwebanything.subdomian.com/s/cnn.com/'],
    'https://cnn.com' : ['http://sub.domain.com/s/cnn.com/', 'http://wwwebanything.subdomian.com/s/cnn.com/'],
    'http://www.cnn.com' : ['http://sub.domain.com/cnn.com/', 'http://wwwebanything.subdomian.com/cnn.com/'],
    'http://cnn.com' : ['http://sub.domain.com/cnn.com/', 'http://wwwebanything.subdomian.com/cnn.com/'],

    'https://www.cnn.com/' : ['http://sub.domain.com/s/cnn.com/', 'http://wwwebanything.subdomian.com/s/cnn.com/'],
    'https://cnn.com/' : ['http://sub.domain.com/s/cnn.com/', 'http://wwwebanything.subdomian.com/s/cnn.com/'],
    'http://www.cnn.com/' : ['http://sub.domain.com/cnn.com/', 'http://wwwebanything.subdomian.com/cnn.com/'],
    'http://cnn.com/' : ['http://sub.domain.com/cnn.com/', 'http://wwwebanything.subdomian.com/cnn.com/'],

    'https://www.cnn.com/new?ffd' : ['http://sub.domain.com/s/cnn.com/new%3Fffd', 'http://wwwebanything.subdomian.com/s/cnn.com/new%3Fffd'],
    'https://cnn.com/new?ffd' : ['http://sub.domain.com/s/cnn.com/new%3Fffd', 'http://wwwebanything.subdomian.com/s/cnn.com/new%3Fffd'],
    'http://www.cnn.com/new?ffd' : ['http://sub.domain.com/cnn.com/new%3Fffd', 'http://wwwebanything.subdomian.com/cnn.com/new%3Fffd'],
    'http://cnn.com/new?ffd' : ['http://sub.domain.com/cnn.com/new%3Fffd', 'http://wwwebanything.subdomian.com/cnn.com/new%3Fffd'],

    'https://www.cnn.com/new/ss?ffd' : ['http://sub.domain.com/s/cnn.com/new%2Fss%3Fffd', 'http://wwwebanything.subdomian.com/s/cnn.com/new%2Fss%3Fffd'],
    'https://cnn.com/new/ss?ffd' : ['http://sub.domain.com/s/cnn.com/new%2Fss%3Fffd', 'http://wwwebanything.subdomian.com/s/cnn.com/new%2Fss%3Fffd'],
    'http://www.cnn.com/new/ss?ffd' : ['http://sub.domain.com/cnn.com/new%2Fss%3Fffd', 'http://wwwebanything.subdomian.com/cnn.com/new%2Fss%3Fffd'],
    'http://cnn.com/new/ss?ffd' : ['http://sub.domain.com/cnn.com/new%2Fss%3Fffd', 'http://wwwebanything.subdomian.com/cnn.com/new%2Fss%3Fffd'],

    'http://www' : ['http://www', 'http://www'],
    'https://www' : ['https://www', 'https://www'],
    'http://www.' : ['http://www.', 'http://www.'],
    'https://www.' : ['https://www.', 'https://www.'],
    'http://www-www' : ['http://www-www', 'http://www-www'],
    'https://www-www' : ['https://www-www', 'https://www-www'],
    'http://www-www.www' : ['http://sub.domain.com/www-www.www/', 'http://wwwebanything.subdomian.com/www-www.www/'],
    'https://www-www.www' : ['http://sub.domain.com/s/www-www.www/', 'http://wwwebanything.subdomian.com/s/www-www.www/'],
    'http://www.www-www.www' : ['http://sub.domain.com/www-www.www/', 'http://wwwebanything.subdomian.com/www-www.www/'],
    'https://www.www-www.www' : ['http://sub.domain.com/s/www-www.www/', 'http://wwwebanything.subdomian.com/s/www-www.www/'],
    
    'http://www1.cnn.com' : ['http://sub.domain.com/www1.cnn.com/', 'http://wwwebanything.subdomian.com/www1.cnn.com/'],
    'https://www1.cnn.com' : ['http://sub.domain.com/s/www1.cnn.com/', 'http://wwwebanything.subdomian.com/s/www1.cnn.com/'],
 
    'http://www1.cnn.com' : ['http://sub.domain.com/www1.cnn.com/', 'http://wwwebanything.subdomian.com/www1.cnn.com/'],
    'https://www1.cnn.com' : ['http://sub.domain.com/s/www1.cnn.com/', 'http://wwwebanything.subdomian.com/s/www1.cnn.com/'],

    'cnn.com' : ['cnn.com', 'cnn.com'],
    'www.cnn.com' : ['www.cnn.com', 'www.cnn.com'],

    '#' : ['#', '#'],
    '#anchor' : ['#anchor', '#anchor'],
    'http://www.some.com/#' : ['http://sub.domain.com/some.com/', 'http://wwwebanything.subdomian.com/some.com/'],
    'http://www.some.com#hash' : ['http://sub.domain.com/some.com/%23hash', 'http://wwwebanything.subdomian.com/some.com/%23hash'],
    'http://www.some.com/#hash' : ['http://sub.domain.com/some.com/%23hash', 'http://wwwebanything.subdomian.com/some.com/%23hash'],
    'http://www.some.com/path/#hash' : ['http://sub.domain.com/some.com/path%2F%23hash', 'http://wwwebanything.subdomian.com/some.com/path%2F%23hash'],
    'http://www.some.com/index.html#hash' : ['http://sub.domain.com/some.com/index.html%23hash', 'http://wwwebanything.subdomian.com/some.com/index.html%23hash'],
    'http://www.some.com/#hash/path' : ['http://sub.domain.com/some.com/%23hash%2Fpath', 'http://wwwebanything.subdomian.com/some.com/%23hash%2Fpath'],
    'http://www.some.com/#/path' : ['http://sub.domain.com/some.com/%23%2Fpath', 'http://wwwebanything.subdomian.com/some.com/%23%2Fpath'],
    'http://www.some.com/#!path' : ['http://sub.domain.com/some.com/%23%21path', 'http://wwwebanything.subdomian.com/some.com/%23%21path'],
    'http://www.some.com/?query=foo#hash' : ['http://sub.domain.com/some.com/%3Fquery%3Dfoo%23hash', 'http://wwwebanything.subdomian.com/some.com/%3Fquery%3Dfoo%23hash'],
    'http://www.some.com/some.com#hash' : ['http://sub.domain.com/some.com/some.com%23hash', 'http://wwwebanything.subdomian.com/some.com/some.com%23hash'],
    'http://www.some.com/path/some.com#hash' : ['http://sub.domain.com/some.com/path%2Fsome.com%23hash', 'http://wwwebanything.subdomian.com/some.com/path%2Fsome.com%23hash']
  };

  var surly = new Surly('AA000015');
  surly.setPanelHost('sub.domain.com');

  for(var url in url2expected) {
    assert.deepEqual([url2expected[url][0]], surly.processMultipleUrls([url]), '[' + url + ']' + ' => ' + '[' + url2expected[url][0] + ']');
  }

  surly.setPanelHost('wwwebanything.subdomian.com');

  for(var url in url2expected) {
    assert.deepEqual([url2expected[url][1]], surly.processMultipleUrls([url]), '[' + url + ']' + ' => ' + '[' + url2expected[url][1] + ']');
  }
});

QUnit.test('process with toolbarId', function(assert) {
  var text = '<a href="http://www.some.com/">link</a>'
    + '<a href="http://www.some.com/thing?test=5&a=5">link</a>'
    + '<a href="https://www.some.com/thing?test=5&a=5">link</a>'
    + '<a href="http://www.some.com/thing/">link</a>'
    + '<a href="https://www.some.com/thing/">link</a>'
    + '<a href="http://www.some.com/thing?test=5&a=5">http://www.some.com/thing?test=5&a=5</a>'
    + '<a href="https://www.some.com/thing?test=5&a=5">https://www.some.com/thing?test=5&a=5</a>'
    + '<a href="http://www.some.com/thing/">http://www.some.com/thing/</a>'
    + '<a href="https://www.some.com/thing/">https://www.some.com/thing/</a>'
    + '<a href="#">link</a>'
    + '<a href="#anchor">link</a>'
    + '<a href="http://www.some.com/#">link</a>'
    + '<a href="http://www.some.com#hash">link</a>'
    + '<a href="http://www.some.com/#hash">link</a>'
    + '<a href="http://www.some.com/path/#hash">link</a>'
    + '<a href="http://www.some.com/index.html#hash">link</a>'
    + '<a href="http://www.some.com/#hash/path">link</a>'
    + '<a href="http://www.some.com/#/path">link</a>'
    + '<a href="http://www.some.com/#!path">link</a>'
    + '<a href="http://www.some.com/?query=foo#hash">link</a>'
    + '<a href="http://www.some.com/some.com#hash">link</a>'
    + '<a href="http://www.some.com/path/some.com#hash">link</a>'
    ;
      
  var expected = '<a href="http://sur.ly/o/some.com/AA000015">link</a>'
    + '<a href="http://sur.ly/o/some.com/thing%3Ftest%3D5%26a%3D5/AA000015">link</a>'
    + '<a href="https://sur.ly/o/some.com/thing%3Ftest%3D5%26a%3D5/AA000015">link</a>'
    + '<a href="http://sur.ly/o/some.com/thing%2F/AA000015">link</a>'
    + '<a href="https://sur.ly/o/some.com/thing%2F/AA000015">link</a>'
    + '<a href="http://sur.ly/o/some.com/thing%3Ftest%3D5%26a%3D5/AA000015">http://www.some.com/thing?test=5&amp;a=5</a>'
    + '<a href="https://sur.ly/o/some.com/thing%3Ftest%3D5%26a%3D5/AA000015">https://www.some.com/thing?test=5&amp;a=5</a>'
    + '<a href="http://sur.ly/o/some.com/thing%2F/AA000015">http://www.some.com/thing/</a>'
    + '<a href="https://sur.ly/o/some.com/thing%2F/AA000015">https://www.some.com/thing/</a>'
    + '<a href="#">link</a>'
    + '<a href="#anchor">link</a>'
    + '<a href="http://sur.ly/o/some.com/AA000015">link</a>'
    + '<a href="http://sur.ly/o/some.com/%23hash/AA000015">link</a>'
    + '<a href="http://sur.ly/o/some.com/%23hash/AA000015">link</a>'
    + '<a href="http://sur.ly/o/some.com/path%2F%23hash/AA000015">link</a>'
    + '<a href="http://sur.ly/o/some.com/index.html%23hash/AA000015">link</a>'
    + '<a href="http://sur.ly/o/some.com/%23hash%2Fpath/AA000015">link</a>'
    + '<a href="http://sur.ly/o/some.com/%23%2Fpath/AA000015">link</a>'
    + '<a href="http://sur.ly/o/some.com/%23%21path/AA000015">link</a>'
    + '<a href="http://sur.ly/o/some.com/%3Fquery%3Dfoo%23hash/AA000015">link</a>'
    + '<a href="http://sur.ly/o/some.com/some.com%23hash/AA000015">link</a>'
    + '<a href="http://sur.ly/o/some.com/path%2Fsome.com%23hash/AA000015">link</a>'
    ;
  
  var fragment = document.createElement('div');
  fragment.innerHTML = text;

  var surly = new Surly('AA000015');  
  surly.process(fragment);

  assert.equal(expected.toLowerCase(), fragment.innerHTML.toLowerCase());
});

QUnit.test('out panel: process with toolbarId', function(assert) {
  var text = '<a href="http://www.some.com/">link</a>'
    + '<a href="http://www.some.com/thing?test=5&a=5">link</a>'
    + '<a href="https://www.some.com/thing?test=5&a=5">link</a>'
    + '<a href="http://www.some.com/thing/">link</a>'
    + '<a href="https://www.some.com/thing/">link</a>'
    + '<a href="http://www.some.com/thing?test=5&a=5">http://www.some.com/thing?test=5&a=5</a>'
    + '<a href="https://www.some.com/thing?test=5&a=5">https://www.some.com/thing?test=5&a=5</a>'
    + '<a href="http://www.some.com/thing/">http://www.some.com/thing/</a>'
    + '<a href="https://www.some.com/thing/">https://www.some.com/thing/</a>'
    + '<a href="#">link</a>'
    + '<a href="#anchor">link</a>'
    + '<a href="http://www.some.com/#">link</a>'
    + '<a href="http://www.some.com#hash">link</a>'
    + '<a href="http://www.some.com/#hash">link</a>'
    + '<a href="http://www.some.com/path/#hash">link</a>'
    + '<a href="http://www.some.com/index.html#hash">link</a>'
    + '<a href="http://www.some.com/#hash/path">link</a>'
    + '<a href="http://www.some.com/#/path">link</a>'
    + '<a href="http://www.some.com/#!path">link</a>'
    + '<a href="http://www.some.com/?query=foo#hash">link</a>'
    + '<a href="http://www.some.com/some.com#hash">link</a>'
    + '<a href="http://www.some.com/path/some.com#hash">link</a>'
    ;
      
  var expected = '<a href="http://sub.domain.com/some.com/">link</a>'
    + '<a href="http://sub.domain.com/some.com/thing%3Ftest%3D5%26a%3D5">link</a>'
    + '<a href="http://sub.domain.com/s/some.com/thing%3Ftest%3D5%26a%3D5">link</a>'
    + '<a href="http://sub.domain.com/some.com/thing%2F">link</a>'
    + '<a href="http://sub.domain.com/s/some.com/thing%2F">link</a>'
    + '<a href="http://sub.domain.com/some.com/thing%3Ftest%3D5%26a%3D5">http://www.some.com/thing?test=5&amp;a=5</a>'
    + '<a href="http://sub.domain.com/s/some.com/thing%3Ftest%3D5%26a%3D5">https://www.some.com/thing?test=5&amp;a=5</a>'
    + '<a href="http://sub.domain.com/some.com/thing%2F">http://www.some.com/thing/</a>'
    + '<a href="http://sub.domain.com/s/some.com/thing%2F">https://www.some.com/thing/</a>'
    + '<a href="#">link</a>'
    + '<a href="#anchor">link</a>'
    + '<a href="http://sub.domain.com/some.com/">link</a>'
    + '<a href="http://sub.domain.com/some.com/%23hash">link</a>'
    + '<a href="http://sub.domain.com/some.com/%23hash">link</a>'
    + '<a href="http://sub.domain.com/some.com/path%2F%23hash">link</a>'
    + '<a href="http://sub.domain.com/some.com/index.html%23hash">link</a>'
    + '<a href="http://sub.domain.com/some.com/%23hash%2Fpath">link</a>'
    + '<a href="http://sub.domain.com/some.com/%23%2Fpath">link</a>'
    + '<a href="http://sub.domain.com/some.com/%23%21path">link</a>'
    + '<a href="http://sub.domain.com/some.com/%3Fquery%3Dfoo%23hash">link</a>'
    + '<a href="http://sub.domain.com/some.com/some.com%23hash">link</a>'
    + '<a href="http://sub.domain.com/some.com/path%2Fsome.com%23hash">link</a>'
    ;

  var fragment = document.createElement('div');
  fragment.innerHTML = text;

  var surly = new Surly('AA000015');
  surly.setPanelHost('sub.domain.com');
  surly.process(fragment);

  assert.equal(expected.toLowerCase(), fragment.innerHTML.toLowerCase());

  expected = '<a href="http://wwwebanything.subdomian.com/some.com/">link</a>'
    + '<a href="http://wwwebanything.subdomian.com/some.com/thing%3Ftest%3D5%26a%3D5">link</a>'
    + '<a href="http://wwwebanything.subdomian.com/s/some.com/thing%3Ftest%3D5%26a%3D5">link</a>'
    + '<a href="http://wwwebanything.subdomian.com/some.com/thing%2F">link</a>'
    + '<a href="http://wwwebanything.subdomian.com/s/some.com/thing%2F">link</a>'
    + '<a href="http://wwwebanything.subdomian.com/some.com/thing%3Ftest%3D5%26a%3D5">http://www.some.com/thing?test=5&amp;a=5</a>'
    + '<a href="http://wwwebanything.subdomian.com/s/some.com/thing%3Ftest%3D5%26a%3D5">https://www.some.com/thing?test=5&amp;a=5</a>'
    + '<a href="http://wwwebanything.subdomian.com/some.com/thing%2F">http://www.some.com/thing/</a>'
    + '<a href="http://wwwebanything.subdomian.com/s/some.com/thing%2F">https://www.some.com/thing/</a>'
    + '<a href="#">link</a>'
    + '<a href="#anchor">link</a>'
    + '<a href="http://wwwebanything.subdomian.com/some.com/">link</a>'
    + '<a href="http://wwwebanything.subdomian.com/some.com/%23hash">link</a>'
    + '<a href="http://wwwebanything.subdomian.com/some.com/%23hash">link</a>'
    + '<a href="http://wwwebanything.subdomian.com/some.com/path%2F%23hash">link</a>'
    + '<a href="http://wwwebanything.subdomian.com/some.com/index.html%23hash">link</a>'
    + '<a href="http://wwwebanything.subdomian.com/some.com/%23hash%2Fpath">link</a>'
    + '<a href="http://wwwebanything.subdomian.com/some.com/%23%2Fpath">link</a>'
    + '<a href="http://wwwebanything.subdomian.com/some.com/%23%21path">link</a>'
    + '<a href="http://wwwebanything.subdomian.com/some.com/%3Fquery%3Dfoo%23hash">link</a>'
    + '<a href="http://wwwebanything.subdomian.com/some.com/some.com%23hash">link</a>'
    + '<a href="http://wwwebanything.subdomian.com/some.com/path%2Fsome.com%23hash">link</a>'
    ;

  fragment.innerHTML = text;

  surly.setPanelHost('wwwebanything.subdomian.com');
  surly.process(fragment);

  assert.equal(expected.toLowerCase(), fragment.innerHTML.toLowerCase());
});

QUnit.test('don‘t replaces panel domains', function(assert) {
  var text = '<a href="http://sur.ly/something.com">link</a>'
    + '<a href="https://sur.ly/something.com">link</a>'
    + '<a href="http://surdotly.com/something.com">link</a>'
    + '<a href="https://surdotly.com/something.com">link</a>';

  var expected = '<a href="http://sur.ly/something.com">link</a>'
    + '<a href="https://sur.ly/something.com">link</a>'
    + '<a href="http://surdotly.com/something.com">link</a>'
    + '<a href="https://surdotly.com/something.com">link</a>';

  var fragment = document.createElement('div');
  fragment.innerHTML = text;

  var surly = new Surly();
  surly.process(fragment);

  assert.equal(expected.toLowerCase(), fragment.innerHTML.toLowerCase());
});

QUnit.test('out panel: don‘t replaces panel subdomain', function(assert) {
  var text = '<a href="http://sub.domain.com">link</a>'
    + '<a href="https://sub.domain.com">link</a>'
    + '<a href="http://sub.domain.com/index.html">link</a>'
    + '<a href="https://sub.domain.com/index.html">link</a>';

  var expected = '<a href="http://sub.domain.com">link</a>'
    + '<a href="https://sub.domain.com">link</a>'
    + '<a href="http://sub.domain.com/index.html">link</a>'
    + '<a href="https://sub.domain.com/index.html">link</a>';

  var fragment = document.createElement('div');
  fragment.innerHTML = text;

  var surly = new Surly();
  surly.setPanelHost('sub.domain.com');
  surly.process(fragment);

  assert.equal(expected.toLowerCase(), fragment.innerHTML.toLowerCase());
});

QUnit.test('replace invalid urls', function(assert) {
  var text = '<a href="sdfgergerg">link</a>'
    + '<a href="http://www.www.www">link</a>'
    + '<a href="https://www.www.www">link</a>'
    + '<a href="http://www.www.">link</a>'
    + '<a href="https://www.www.">link</a>'
    + '<a href="http://www.">link</a>'
    + '<a href="https://www.">link</a>'
    + '<a href="http://www">link</a>'
    + '<a href="https://www">link</a>'
    + '<a href="http://www-www">link</a>'
    + '<a href="https://www-www">link</a>'
    + '<a href="http://www-www.www">link</a>'
    + '<a href="https://www-www.www">link</a>'
    + '<a href="http://www.www-www.www">link</a>'
    + '<a href="https://www.www-www.www">link</a>'
    + '<a href="http://">link</a>'
    + '<a href="https://">link</a>'
    + '<a href="http://www.www.www">http://www.www.www</a>'
    + '<a href="https://www.www.www">https://www.www.www</a>'
    + '<a href="http://www.www.">http://www.www.</a>'
    + '<a href="https://www.www.">https://www.www.</a>'
    + '<a href="http://www.">http://www.</a>'
    + '<a href="https://www.">https://www.</a>'
    + '<a href="http://www">http://www</a>'
    + '<a href="https://www">https://www</a>'
    + '<a href="http://www-www">http://www-www</a>'
    + '<a href="https://www-www">https://www-www</a>'
    + '<a href="http://www-www.www">http://www-www.www</a>'
    + '<a href="https://www-www.www">https://www-www.www</a>'
    + '<a href="http://www.www-www.www">http://www.www-www.www</a>'
    + '<a href="https://www.www-www.www">https://www.www-www.www</a>'
    + '<a href="http://www.www-www.www/">http://www.www-www.www/</a>'
    + '<a href="https://www.www-www.www/">https://www.www-www.www/</a>'
    + '<a href="http://">http://</a>'
    + '<a href="https://">https://</a>'
    + '<a>link</a>'
    + '<a href="#">link</a>'
    + '<a href="#anchor">link</a>'
    + '<a href="javascript:void(0)">link</a>'
    ;

  var expected = '<a href="sdfgergerg">link</a>'
    + '<a href="http://sur.ly/o/www.www/AA000015">link</a>'
    + '<a href="https://sur.ly/o/www.www/AA000015">link</a>'
    + '<a href="http://sur.ly/o/www./AA000015">link</a>'
    + '<a href="https://sur.ly/o/www./AA000015">link</a>'
    + '<a href="http://www.">link</a>'
    + '<a href="https://www.">link</a>'
    + '<a href="http://www">link</a>'
    + '<a href="https://www">link</a>'
    + '<a href="http://www-www">link</a>'
    + '<a href="https://www-www">link</a>'
    + '<a href="http://sur.ly/o/www-www.www/AA000015">link</a>'
    + '<a href="https://sur.ly/o/www-www.www/AA000015">link</a>'
    + '<a href="http://sur.ly/o/www-www.www/AA000015">link</a>'
    + '<a href="https://sur.ly/o/www-www.www/AA000015">link</a>'
    + '<a href="http://">link</a>'
    + '<a href="https://">link</a>'
    + '<a href="http://sur.ly/o/www.www/AA000015">http://www.www.www</a>'
    + '<a href="https://sur.ly/o/www.www/AA000015">https://www.www.www</a>'
    + '<a href="http://sur.ly/o/www./AA000015">http://www.www.</a>'
    + '<a href="https://sur.ly/o/www./AA000015">https://www.www.</a>'
    + '<a href="http://www.">http://www.</a>'
    + '<a href="https://www.">https://www.</a>'
    + '<a href="http://www">http://www</a>'
    + '<a href="https://www">https://www</a>'
    + '<a href="http://www-www">http://www-www</a>'
    + '<a href="https://www-www">https://www-www</a>'
    + '<a href="http://sur.ly/o/www-www.www/AA000015">http://www-www.www</a>'
    + '<a href="https://sur.ly/o/www-www.www/AA000015">https://www-www.www</a>'
    + '<a href="http://sur.ly/o/www-www.www/AA000015">http://www.www-www.www</a>'
    + '<a href="https://sur.ly/o/www-www.www/AA000015">https://www.www-www.www</a>'
    + '<a href="http://sur.ly/o/www-www.www/AA000015">http://www.www-www.www/</a>'
    + '<a href="https://sur.ly/o/www-www.www/AA000015">https://www.www-www.www/</a>'
    + '<a href="http://">http://</a>'
    + '<a href="https://">https://</a>'
    + '<a>link</a>'
    + '<a href="#">link</a>'
    + '<a href="#anchor">link</a>'
    + '<a href="javascript:void(0)">link</a>'
    ;

  var fragment = document.createElement('div');
  fragment.innerHTML = text;

  var surly = new Surly();
  surly.process(fragment);

  assert.equal(expected.toLowerCase(), fragment.innerHTML.toLowerCase());
});

QUnit.test('replace invalid urls with toolbarId', function(assert) {
  var text = '<a href="sdfgergerg">link</a>'
    + '<a href="http://www.www.www">link</a>'
    + '<a href="https://www.www.www">link</a>'
    + '<a href="http://www.www.">link</a>'
    + '<a href="https://www.www.">link</a>'
    + '<a href="http://www.">link</a>'
    + '<a href="https://www.">link</a>'
    + '<a href="http://www">link</a>'
    + '<a href="https://www">link</a>'
    + '<a href="http://www-www">link</a>'
    + '<a href="https://www-www">link</a>'
    + '<a href="http://www-www.www">link</a>'
    + '<a href="https://www-www.www">link</a>'
    + '<a href="http://www.www-www.www">link</a>'
    + '<a href="https://www.www-www.www">link</a>'
    + '<a href="http://">link</a>'
    + '<a href="https://">link</a>'
    + '<a href="http://www.www.www">http://www.www.www</a>'
    + '<a href="https://www.www.www">https://www.www.www</a>'
    + '<a href="http://www.www.">http://www.www.</a>'
    + '<a href="https://www.www.">https://www.www.</a>'
    + '<a href="http://www.">http://www.</a>'
    + '<a href="https://www.">https://www.</a>'
    + '<a href="http://www">http://www</a>'
    + '<a href="https://www">https://www</a>'
    + '<a href="http://www-www">http://www-www</a>'
    + '<a href="https://www-www">https://www-www</a>'
    + '<a href="http://www-www.www">http://www-www.www</a>'
    + '<a href="https://www-www.www">https://www-www.www</a>'
    + '<a href="http://www.www-www.www">http://www.www-www.www</a>'
    + '<a href="https://www.www-www.www">https://www.www-www.www</a>'
    + '<a href="http://www.www-www.www/">http://www.www-www.www/</a>'
    + '<a href="https://www.www-www.www/">https://www.www-www.www/</a>'
    + '<a href="http://">http://</a>'
    + '<a href="https://">https://</a>'
    + '<a>link</a>'
    + '<a href="#">link</a>'
    + '<a href="#anchor">link</a>'
    + '<a href="javascript:void(0)">link</a>'
    ;

  var expected = '<a href="sdfgergerg">link</a>'
    + '<a href="http://sur.ly/o/www.www/AA000015">link</a>'
    + '<a href="https://sur.ly/o/www.www/AA000015">link</a>'
    + '<a href="http://sur.ly/o/www./AA000015">link</a>'
    + '<a href="https://sur.ly/o/www./AA000015">link</a>'
    + '<a href="http://www.">link</a>'
    + '<a href="https://www.">link</a>'
    + '<a href="http://www">link</a>'
    + '<a href="https://www">link</a>'
    + '<a href="http://www-www">link</a>'
    + '<a href="https://www-www">link</a>'
    + '<a href="http://sur.ly/o/www-www.www/AA000015">link</a>'
    + '<a href="https://sur.ly/o/www-www.www/AA000015">link</a>'
    + '<a href="http://sur.ly/o/www-www.www/AA000015">link</a>'
    + '<a href="https://sur.ly/o/www-www.www/AA000015">link</a>'
    + '<a href="http://">link</a>'
    + '<a href="https://">link</a>'
    + '<a href="http://sur.ly/o/www.www/AA000015">http://www.www.www</a>'
    + '<a href="https://sur.ly/o/www.www/AA000015">https://www.www.www</a>'
    + '<a href="http://sur.ly/o/www./AA000015">http://www.www.</a>'
    + '<a href="https://sur.ly/o/www./AA000015">https://www.www.</a>'
    + '<a href="http://www.">http://www.</a>'
    + '<a href="https://www.">https://www.</a>'
    + '<a href="http://www">http://www</a>'
    + '<a href="https://www">https://www</a>'
    + '<a href="http://www-www">http://www-www</a>'
    + '<a href="https://www-www">https://www-www</a>'
    + '<a href="http://sur.ly/o/www-www.www/AA000015">http://www-www.www</a>'
    + '<a href="https://sur.ly/o/www-www.www/AA000015">https://www-www.www</a>'
    + '<a href="http://sur.ly/o/www-www.www/AA000015">http://www.www-www.www</a>'
    + '<a href="https://sur.ly/o/www-www.www/AA000015">https://www.www-www.www</a>'
    + '<a href="http://sur.ly/o/www-www.www/AA000015">http://www.www-www.www/</a>'
    + '<a href="https://sur.ly/o/www-www.www/AA000015">https://www.www-www.www/</a>'
    + '<a href="http://">http://</a>'
    + '<a href="https://">https://</a>'
    + '<a>link</a>'
    + '<a href="#">link</a>'
    + '<a href="#anchor">link</a>'
    + '<a href="javascript:void(0)">link</a>'
    ;

  var fragment = document.createElement('div');
  fragment.innerHTML = text;

  var surly = new Surly('AA000015');
  surly.process(fragment);

  assert.equal(expected.toLowerCase(), fragment.innerHTML.toLowerCase());
});

QUnit.test('attributes', function(assert) {
  var text = '<a id="id" class="class" title="title" rel="details" target="_blank" href="http://www.www.www">link</a>';

  var expected = [
    {'id':'id', 'class':'class', 'title':'title', 'rel':'details', 'target':'_blank', 'href':'http://sur.ly/o/www.www/AA000015'}
  ];

  var fragment = document.createElement('div');
  fragment.innerHTML = text;

  var surly = new Surly();
  surly.process(fragment);

  var links = fragment.getElementsByTagName('A');

  for(var i = 0; i < links.length; i++) {
    assert.equal(expected[i]['id'], links[i].getAttribute('id'));
    assert.equal(expected[i]['class'], links[i].getAttribute('class'));
    assert.equal(expected[i]['title'], links[i].getAttribute('title'));
    assert.equal(expected[i]['rel'], links[i].getAttribute('rel'));
    assert.equal(expected[i]['target'], links[i].getAttribute('target'));
    assert.equal(expected[i]['href'], links[i].getAttribute('href'));
  }
});

QUnit.test('don‘t replace relative links', function(assert) {
  var text = '<a href="/somepage">link</a>'
    + '<a href="/index.php?q=node/3">link</a>';

  var fragment = document.createElement('div');
  fragment.innerHTML = text;

  var surly = new Surly();
  surly.process(fragment);

  assert.equal(text.toLowerCase(), fragment.innerHTML.toLowerCase());
});

QUnit.test('domain whitelist', function(assert) {
  var text = '<a href="http://le.com">link</a>'
    + '<a href="https://le.com">link</a>'
    + '<a href="http://le.com#something">link</a>'
    + '<a href="https://le.com#something">link</a>'
    + '<a href="http://lE.cOm">link</a>'
    + '<a href="https://lE.cOm">link</a>'
    + '<a href="http://sub.le.com">link</a>'
    + '<a href="https://sub.le.com">link</a>'
    + '<a href="http://www.le.com">link</a>'
    + '<a href="https://www.le.com">link</a>'
    + '<a href="http://le.com/something">link</a>'
    + '<a href="https://le.com/something">link</a>'
    + '<a href="http://www.le.com/something">link</a>'
    + '<a href="https://www.le.com/something">link</a>'
    + '<a href="http://www.google.com/something">link</a>'
    + '<a href="https://www.google.com/something">link</a>'
    + '<a href="http://google.com/something">link</a>'
    + '<a href="https://google.com/something">link</a>'
    + '<a href="http://www.google.com">link</a>'
    + '<a href="https://www.google.com">link</a>'
    + '<a href="http://google.com">link</a>'
    + '<a href="https://google.com">link</a>';

  var expected = ['http://le.com',
    'https://le.com',
    'http://le.com#something',
    'https://le.com#something',
    'http://lE.cOm',
    'https://lE.cOm',
    'http://sub.le.com',
    'https://sub.le.com',
    'http://www.le.com',
    'https://www.le.com',
    'http://le.com/something',
    'https://le.com/something',
    'http://www.le.com/something',
    'https://www.le.com/something',
    'http://sur.ly/o/google.com/something/AA000015',
    'https://sur.ly/o/google.com/something/AA000015',
    'http://sur.ly/o/google.com/something/AA000015',
    'https://sur.ly/o/google.com/something/AA000015',
    'http://sur.ly/o/google.com/AA000015',
    'https://sur.ly/o/google.com/AA000015',
    'http://sur.ly/o/google.com/AA000015',
    'https://sur.ly/o/google.com/AA000015'
  ];

  var fragment = document.createElement('div');
  fragment.innerHTML = text;

  var surly = new Surly();
  surly.whiteList('lE.com');
  surly.process(fragment);

  var links = fragment.getElementsByTagName('A');

  for(var i = 0; i < links.length; i++) {
    assert.equal(expected[i], links[i].getAttribute('href'));
  }

  fragment.innerHTML = text;
  surly.process(fragment);

  surly = new Surly();
  surly.whiteList('www.le.com');


  links = fragment.getElementsByTagName('A');

  for(var i = 0; i < links.length; i++) {
    assert.equal(expected[i], links[i].getAttribute('href'));
  }
});

QUnit.test('domain whitelist with http', function(assert) {
  var text = '<a href="http://test.com">test</a>';
  var expected = '<a href="http://test.com">test</a>';

  var fragment = document.createElement('div');
  fragment.innerHTML = text;

  var surly = new Surly();
  surly.whiteList('http://test.com');
  surly.process(fragment);

  assert.equal(expected.toLowerCase(), fragment.innerHTML.toLowerCase());
});

QUnit.test('domain whitelist with https', function(assert) {
  var text = '<a href="https://test.com">test</a>';
  var expected = '<a href="https://test.com">test</a>';

  var fragment = document.createElement('div');
  fragment.innerHTML = text;

  var surly = new Surly();
  surly.whiteList('https://test.com');
  surly.process(fragment);

  assert.equal(expected.toLowerCase(), fragment.innerHTML.toLowerCase());
});

QUnit.test('domain whitelist with www', function(assert) {
  var text = '<a href="http://wwwebsite.com">link</a>';
  var expected = '<a href="http://wwwebsite.com">link</a>';

  var fragment = document.createElement('div');
  fragment.innerHTML = text;

  var surly = new Surly();
  surly.whiteList('http://wwwebsite.com');
  surly.process(fragment);

  assert.equal(expected.toLowerCase(), fragment.innerHTML.toLowerCase());

  text = '<a href="http://wwwebanything.subdomain.com">link</a>';
  expected = '<a href="http://wwwebanything.subdomain.com">link</a>';

  fragment.innerHTML = text;

  surly.whiteList('http://wwwebanything.subdomain.com');
  surly.process(fragment);

  assert.equal(expected.toLowerCase(), fragment.innerHTML.toLowerCase());  
});